using System;
using System.Xml;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class holds information returned by remote VistALink server 
	/// (M listener, .NET listener) in response to VistALink setup and 
	/// introductory text request. 
	/// </summary>
	public class RemoteServerSetupAndIntroInfo
	{	
		private string _introText;
		private string _serverName; 
		private string _volume; 
		private string _uci;
		private string _device;
		private int _numberOfAttempts;
		private Timeout _timeout;

		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_INTRO_TEXT_NODE_NAME = "IntroText";
		private const string XMLCONSTS_LINEBREAK_CHARACTER = "<BR>";
		private const string XMLCONSTS_SETUPINFO_NODE_NAME = "SetupInfo";
		private const string XMLCONSTS_SERVER_NAME_SI_ATTRIBUTE_NAME = "serverName";
		private const string XMLCONSTS_VOLUME_SI_ATTRIBUTE_NAME = "volume";
		private const string XMLCONSTS_UCI_SI_ATTRIBUTE_NAME = "uci";
		private const string XMLCONSTS_DEVICE_SI_ATTRIBUTE_NAME = "device";
		private const string XMLCONSTS_NUMBER_OF_ATTEMPTS_SI_ATTRIBUTE_NAME = "numberAttempts";
		private const string XMLCONSTS_TIMEOUT_SI_ATTRIBUTE_NAME = "dtime";

		/// <summary>
		/// The only constructor. All parameters are optional.
		/// </summary>
		/// <param name="serverName">Server name.</param>
		/// <param name="volume">VistA volume.</param>
		/// <param name="uci">VistA UCI.</param>
		/// <param name="device">Device specification.</param>
		/// <param name="numberOfAttempts">Number of allowed logon attempts.</param>
		/// <param name="timeout">Server timeout before disconnect.</param>
		/// <param name="introText">Server sign-in text. </param>
		public RemoteServerSetupAndIntroInfo( string serverName, string volume, string uci, string device, int numberOfAttempts, Timeout timeout, string introText )
		{
			if( serverName == null )
				throw( new ArgumentNullException( "serverName" ) );

			if( volume == null )
				throw( new ArgumentNullException( "volume" ) );

			if( uci == null )
				throw( new ArgumentNullException( "uci" ) );

			if( device == null )
				throw( new ArgumentNullException( "device" ) );

			if( numberOfAttempts < 0 )
				throw( new ArgumentOutOfRangeException( SR.Exceptions.InvalidNumberOfAttempts( numberOfAttempts.ToString() ) ) );

			if( timeout == null )
				throw( new ArgumentNullException( "timeout" ) );

			if( introText == null )
				throw( new ArgumentNullException( "introText" ) );

			_serverName = serverName;
			_volume = volume;
			_uci = uci;
			_device = device;
			_numberOfAttempts = numberOfAttempts;
			_timeout = timeout;
			_introText = introText;
		}

		/// <summary>
		/// XML deserialization factory method parsing supplied remote serve setup and 
		/// intro info root XML element and returning remote server setup and intro
		/// info object created from it.
		/// </summary>
		/// <param name="rootElement">
		///		XML element containing remote server setup and intro 
		///		info XML nodes among its children.
		///	</param>
		/// <returns>Remote server setup and intro info object deserialized from XML.</returns>
		public static RemoteServerSetupAndIntroInfo Parse( XmlElement rootElement )
		{
			if( rootElement == null )
				throw( new ArgumentNullException( "rootElement" ) );

			XmlElement _setupInfo = XmlUtility.ParseGetRequiredElementByUniqueTagName( rootElement, XMLCONSTS_SETUPINFO_NODE_NAME );
			XmlElement _introTextElement = XmlUtility.ParseGetRequiredElementByUniqueTagName( rootElement, XMLCONSTS_INTRO_TEXT_NODE_NAME );
			
			try
			{			
				return new RemoteServerSetupAndIntroInfo( 
					XmlUtility.ParseGetRequiredAttributeValue( _setupInfo, XMLCONSTS_SERVER_NAME_SI_ATTRIBUTE_NAME ), 
					XmlUtility.ParseGetRequiredAttributeValue( _setupInfo, XMLCONSTS_VOLUME_SI_ATTRIBUTE_NAME ), 
					XmlUtility.ParseGetRequiredAttributeValue( _setupInfo, XMLCONSTS_UCI_SI_ATTRIBUTE_NAME ), 
					XmlUtility.ParseGetRequiredAttributeValue( _setupInfo, XMLCONSTS_DEVICE_SI_ATTRIBUTE_NAME ), 
					Int32.Parse( XmlUtility.ParseGetRequiredAttributeValue( _setupInfo, XMLCONSTS_NUMBER_OF_ATTEMPTS_SI_ATTRIBUTE_NAME ) ), 
					Timeout.ParseStringInSeconds( XmlUtility.ParseGetRequiredAttributeValue( _setupInfo, XMLCONSTS_TIMEOUT_SI_ATTRIBUTE_NAME ) ), 
					_introTextElement.InnerText.Replace( XMLCONSTS_LINEBREAK_CHARACTER, Environment.NewLine ) );
			}
			catch( FlatMediumParseException )
			{
				throw;
			}
			catch( Exception xcp ) // have to catch exception here because Int32.Parse throws 3 different exceptions. 
			{
				// Rethrowing if this is not exception caused by Int32.Parse or RemoteServerInfo constructor
				if( !( xcp is FormatException || xcp is OverflowException || xcp is ArgumentException ) )
					throw;

				throw( new XmlParseException( SR.Exceptions.XmlMessageParseCausedExceptionInConstructor( 
					typeof(RemoteServerSetupAndIntroInfo).Name, xcp.Message ), xcp ) );
			}
		}

		/// <summary>
		/// XML serialization method writing out remote server 
		/// setup and intro info to suppied XML writer.
		/// </summary>
		/// <param name="writer">XmlWriter to write to.</param>
		public void WriteSetupAndIntroInfoToXml( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_SETUPINFO_NODE_NAME );
			
			writer.WriteAttributeString( XMLCONSTS_SERVER_NAME_SI_ATTRIBUTE_NAME, _serverName );
			writer.WriteAttributeString( XMLCONSTS_VOLUME_SI_ATTRIBUTE_NAME, _volume );
			writer.WriteAttributeString( XMLCONSTS_UCI_SI_ATTRIBUTE_NAME, _uci );
			writer.WriteAttributeString( XMLCONSTS_DEVICE_SI_ATTRIBUTE_NAME, _device );
			writer.WriteAttributeString( XMLCONSTS_NUMBER_OF_ATTEMPTS_SI_ATTRIBUTE_NAME, _numberOfAttempts.ToString() );
			writer.WriteAttributeString( XMLCONSTS_TIMEOUT_SI_ATTRIBUTE_NAME, _timeout.ToStringSeconds() );				

			writer.WriteEndElement();

			XmlUtility.GenWriteElementWithCData( writer, XMLCONSTS_INTRO_TEXT_NODE_NAME, _introText.Replace( Environment.NewLine, XMLCONSTS_LINEBREAK_CHARACTER ) );
		}

		/// <summary>
		/// Server name.
		/// </summary>
		public string ServerName 
		{
			get
			{
				return _serverName;
			}
		}

		/// <summary>
		/// VistA volume.
		/// </summary>
		public string Volume
		{
			get
			{
				return _volume;
			}
		}
		
		/// <summary>
		/// VistA UCI.
		/// </summary>
		public string Uci
		{
			get
			{
				return _uci;
			}
		}

		/// <summary>
		/// Device specification.
		/// </summary>
		public string Device
		{
			get
			{
				return _device;
			}
		}

		/// <summary>
		/// Number of allowed logon attempts.
		/// </summary>
		public int NumberOfAttempts
		{
			get
			{
				return _numberOfAttempts;
			}
		}

		/// <summary>
		/// Server timeout before disconnect.
		/// </summary>
		public Timeout Timeout
		{
			get
			{
				return _timeout;
			}
		}

		/// <summary>
		/// Server sign-in text. 
		/// </summary>
		public string IntroText
		{
			get
			{
				return _introText;
			}
		}
	}
}
